import org.codehaus.groovy.grails.plugins.springsecurity.Secured
import org.codehaus.groovy.grails.commons.ConfigurationHolder

@Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager'])
class InventoryItemDetailedController extends BaseController {

    def filterService
    def exportService
    def inventoryItemService
    def inventoryMovementService

    // the delete, save and update actions only accept POST requests
    static allowedMethods = [delete:'POST', save:'POST', update:'POST', useInventoryItem:'POST']

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def index = { redirect(action:search, params:params) }

    /**
    * Set session.inventoryItemSearchParamsMax
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def setSearchParamsMax = {
        def max = 1000
        if(params.newMax.isInteger()) {
            def i = params.newMax.toInteger()
            if(i > 0 && i <= max)
                session.inventoryItemSearchParamsMax = params.newMax
            if(i > max)
                session.inventoryItemSearchParamsMax = max
        }
        forward(action: 'search', params: params)
    }

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def search = {

        if(session.inventoryItemSearchParamsMax)
            params.max = session.inventoryItemSearchParamsMax

        params.max = Math.min( params.max ? params.max.toInteger() : 10,  100)

        def inventoryItemInstanceList = []
        def inventoryItemInstanceTotal
        def filterParams = [:]

        // Quick Search:
        if(!params.filter) {
            inventoryItemInstanceList = InventoryItem.list( params )
            inventoryItemInstanceTotal = InventoryItem.count()
            filterParams = params
        }
        else {
        // filterPane:
            inventoryItemInstanceList = filterService.filter( params, InventoryItem )
            inventoryItemInstanceTotal = filterService.count( params, InventoryItem )
            filterParams = com.zeddware.grails.plugins.filterpane.FilterUtils.extractFilterParams(params)
        }

        // export plugin:
        if(params?.format && params.format != "html") {

            def dateFmt = { date ->
                formatDate(format: "EEE, dd-MMM-yyyy", date: date)
            }
            String title = "Inventory List."

            response.contentType = ConfigurationHolder.config.grails.mime.types[params.format]
            response.setHeader("Content-disposition", "attachment; filename=Inventory.${params.extension}")
            List fields = ["name",
                                "description",
                                "unitsInStock",
                                "unitOfMeasure",
                                "inventoryLocation",
                                "inventoryLocation.inventoryStore"]
            Map labels = ["name": "Name",
                                "description": "Description",
                                "unitsInStock":"In Stock",
                                "unitOfMeasure": "UOM",
                                "inventoryLocation": "Location",
                                "inventoryLocation.inventoryStore": "Store"]

            Map formatters = [:]
            Map parameters = [title: title, separator: ","]

            exportService.export(params.format,
                                                response.outputStream,
                                                inventoryItemInstanceList.sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) },
                                                fields,
                                                labels,
                                                formatters,
                                                parameters)
        }

        // Add some basic params to filterParams.
        filterParams.max = params.max
        filterParams.offset = params.offset?.toInteger() ?: 0
        filterParams.sort = params.sort ?: "id"
        filterParams.order = params.order ?: "desc"

        return[ inventoryItemInstanceList: inventoryItemInstanceList,
                inventoryItemInstanceTotal: inventoryItemInstanceTotal,
                filterParams: filterParams ]
    } // end search()

    /**
    * Simply assigns a passed in task id to a session variable and redirects to search.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def findInventoryItemForMovement = {
        if(!params.task?.id) {
            flash.message = "No task id supplied, please select a task then the inventory tab."
            redirect(controller: "taskDetailed", action: "search")
            return
        }

        session.inventoryMovementTaskId = params.task.id
        flash.message = "Please find and then select the inventory item."
        redirect(action: search)
    }

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def show = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Show)
            params.action='show'

        if(!InventoryItem.exists(params.id)) {
            flash.message = "InventoryItem not found with id ${params.id}"
            redirect(action:search)
            return
        }

        def result = inventoryItemService.prepareShowData(params)

        if(result.error) {
            flash.message = "Could not to prepare the data to show item with id: ${params.id}."
            redirect(action:search)
            return
        }

        def model = [ inventoryItemInstance: result.inventoryItemInstance,
                                inventoryMovementList: result.inventoryMovementList,
                                inventoryMovementListTotal: result.inventoryMovementListTotal,
                                inventoryMovementListMax: result.inventoryMovementListMax,
                                showTab: result.showTab]

        if(session.inventoryMovementTaskId) {
            model.inventoryMovementInstance = new InventoryMovement()
            model.inventoryMovementInstance.task = Task.get(session.inventoryMovementTaskId)
            model.inventoryMovementInstance.quantity = 1
        }

        return model
    }

    def delete = {
        def inventoryItemInstance = InventoryItem.get( params.id )
        if(inventoryItemInstance) {
            try {
                inventoryItemInstance.delete(flush:true)
                flash.message = "InventoryItem ${params.id} deleted"
                redirect(action:search)
            }
            catch(org.springframework.dao.DataIntegrityViolationException e) {
                flash.message = "InventoryItem ${params.id} could not be deleted"
                redirect(action:show,id:params.id)
            }
        }
        else {
            flash.message = "InventoryItem not found with id ${params.id}"
            redirect(action:search)
        }
    }

    def edit = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Edit)
            params.action='edit'

        def inventoryItemInstance = InventoryItem.get( params.id )

        if(!inventoryItemInstance) {
            flash.message = "InventoryItem not found with id ${params.id}"
            redirect(action:search)
        }
        else {
            return [ inventoryItemInstance : inventoryItemInstance ]
        }
    }

    def update = {
        def inventoryItemInstance = InventoryItem.get( params.id )
        if(inventoryItemInstance) {
            if(params.version) {
                def version = params.version.toLong()
                if(inventoryItemInstance.version > version) {
                    
                    inventoryItemInstance.errors.rejectValue("version", "default.optimistic.locking.failure")
                    render(view:'edit',model:[inventoryItemInstance:inventoryItemInstance])
                    return
                }
            }
            inventoryItemInstance.properties = params
            if(!inventoryItemInstance.hasErrors() && inventoryItemInstance.save(flush: true)) {
                flash.message = "InventoryItem ${params.id} updated"
                redirect(action:show,id:inventoryItemInstance.id)
            }
            else {
                render(view:'edit',model:[inventoryItemInstance:inventoryItemInstance])
            }
        }
        else {
            flash.message = "InventoryItem not found with id ${params.id}"
            redirect(action:search)
        }
    }

    def create = {
        def inventoryItemInstance = new InventoryItem()
        inventoryItemInstance.properties = params
        return ['inventoryItemInstance':inventoryItemInstance]
    }

    def save = {
        def inventoryItemInstance = new InventoryItem(params)
        if(!inventoryItemInstance.hasErrors() && inventoryItemInstance.save(flush: true)) {
            flash.message = "InventoryItem ${inventoryItemInstance.id} created"
            redirect(action:show,id:inventoryItemInstance.id)
        }
        else {
            render(view:'create',model:[inventoryItemInstance:inventoryItemInstance])
        }
    }

    /**
    * Handles the use inventory item form submit in the show view.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_InventoryManager', 'ROLE_InventoryUser'])
    def useInventoryItem = {

        params.inventoryMovementType = InventoryMovementType.get(1) // Set type to "Used".
        def result = inventoryMovementService.move(params)

        if(!result.error) {
            flash.message = "Inventory Movement for ${result.inventoryMovementInstance.inventoryItem.name.encodeAsHTML()} created."
            redirect(controller:"taskDetailed", action:"show", id: result.taskId)
        }
        else {
            if(result.inventoryMovementInstance) {
                def p = [:]
                p.id = result.inventoryMovementInstance.inventoryItem?.id
                def r = inventoryItemService.prepareShowData(p)

                def model = [ inventoryItemInstance: r.inventoryItemInstance,
                                        inventoryMovementList: r.inventoryMovementList,
                                        inventoryMovementListTotal: r.inventoryMovementListTotal,
                                        inventoryMovementListMax: r.inventoryMovementListMax,
                                        showTab: r.showTab]

                model.inventoryMovementInstance = result.inventoryMovementInstance

                render(view: 'show', model: model)
            }
            else {
                flash.message = "Could not create inventory movement."
                redirect(action:"search")
            }

        }
    }

}
